package tech.espublico.pades.server.rest.jetty;

import java.util.ArrayList;
import java.util.List;

import org.eclipse.jetty.server.Connector;
import org.eclipse.jetty.server.HttpConfiguration;
import org.eclipse.jetty.server.HttpConnectionFactory;
import org.eclipse.jetty.server.SecureRequestCustomizer;
import org.eclipse.jetty.server.Server;
import org.eclipse.jetty.server.ServerConnector;
import org.eclipse.jetty.server.SslConnectionFactory;
import org.eclipse.jetty.util.ssl.SslContextFactory;
import org.eclipse.jetty.util.thread.QueuedThreadPool;
import org.eclipse.jetty.util.thread.ThreadPool;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import tech.espublico.pades.server.services.ConfigService;
import tech.espublico.pades.server.services.ExtProperties;
import tech.espublico.pades.server.services.PropertyValidatorException;

public class JettyServerFactory implements spark.embeddedserver.jetty.JettyServerFactory {

	private static final Logger log = LoggerFactory.getLogger(JettyServerFactory.class);

	private final boolean stopAtShutdown;
	private final int stopTimeout;

	private final int threadPoolMinThreads;
	private final int threadPoolMaxThreads;

	public JettyServerFactory() throws PropertyValidatorException {
		this.stopAtShutdown = ConfigService.instance().getBool("jetty.server.stop-at-shutdown", true);
		this.stopTimeout = ConfigService.instance().getInt("jetty.server.graceful-shutdown");

		this.threadPoolMinThreads = ConfigService.instance().getInt("jetty.thread-pool.min-threads");
		this.threadPoolMaxThreads = ConfigService.instance().getInt("jetty.thread-pool.max-threads");
	}

	@Override
	public Server create(int maxThreads, int minThreads, int threadTimeoutMillis) {

		if (minThreads == -1 || maxThreads == -1) {
			minThreads = this.threadPoolMinThreads;
			maxThreads = this.threadPoolMaxThreads;
		}

		QueuedThreadPool threadPool = new QueuedThreadPool();
		threadPool.setIdleTimeout(threadTimeoutMillis);
		threadPool.setMinThreads(minThreads);
		threadPool.setMaxThreads(maxThreads);
		return this.create(threadPool);
	}

	@Override
	public Server create(ThreadPool threadPool) {
		Server server = new Server(threadPool);
		server.setStopAtShutdown(this.stopAtShutdown);
		server.setStopTimeout(this.stopTimeout);
		server.setConnectors(this.buildConnectors(server));
		return server;
	}

	private Connector[] buildConnectors(Server server) {
		try {
			String[] connectorsList = ConfigService.instance().getStringArray("jetty.connector.list", new String[0]);
			if (connectorsList.length == 0) {
				return new Connector[] { this.buildConnector(server) };
			} else {
				List<Connector> connectors = new ArrayList<>();

				for (String connectorConfigKey : connectorsList) {
					connectors.add(this.buildConnector(connectorConfigKey, server));
				}
				return connectors.toArray(new Connector[0]);
			}
		} catch (PropertyValidatorException e) {
			throw new RuntimeException(e);
		}
	}

	/**
	 * Crea un conector cuando no se configura una lista de conectores y solo se pretende usar un único conector.
	 *
	 * @return connector configurado segun las propiedades asociadas a la clave recibida
	 *
	 * @throws PropertyValidatorException
	 */
	private Connector buildConnector(Server server) throws PropertyValidatorException {
		ExtProperties extProps = ConfigService.instance().subProperties("jetty.connector.", "");
		return this.buildConnector(extProps, server);
	}

	/**
	 * Creacion y configuracion de un conector unico a partir de la clave del conector
	 *
	 * @param connectorKey
	 * 		clave del conector a configurar segun la lista de conectores
	 *
	 * @return connector configurado segun las propiedades asociadas a la clave recibida
	 *
	 * @throws PropertyValidatorException
	 */
	private Connector buildConnector(String connectorKey, Server server) throws PropertyValidatorException {
		ExtProperties extProps = ConfigService.instance().subProperties("jetty." + connectorKey + ".connector.", "");
		return this.buildConnector(extProps, server);
	}

	/**
	 * Creacion y configuracion de un conector unico a partir de una configuración.
	 *
	 * @param extProps
	 * 		propiedades del conector.
	 *
	 * @return connector configurado segun las propiedades asociadas a la clave recibida
	 *
	 * @throws PropertyValidatorException
	 */
	private Connector buildConnector(ExtProperties extProps, Server server) throws PropertyValidatorException {
		ServerConnector connector;
		int port = extProps.getInt("port");
		int selectors = extProps.getInt("selectors");
		int acceptors = extProps.getInt("acceptors");

		log.debug("Adding server connector in port {}", port);

		// HTTP Config
		HttpConfiguration httpConf = new HttpConfiguration();
		httpConf.setRequestHeaderSize(extProps.getInt("request-header-size"));
		httpConf.setResponseHeaderSize(extProps.getInt("response-header-size"));
		httpConf.setSendServerVersion(extProps.getBool("send-server-version"));
		httpConf.setSendDateHeader(extProps.getBool("send-date-header"));

		String scheme = extProps.getString("scheme");
		if ("https".equals(scheme)) {
			httpConf.setSecurePort(port);
			httpConf.setSecureScheme("https");
			httpConf.addCustomizer(new SecureRequestCustomizer());

			// Setup SSL
			SslContextFactory sslContextFactory = new SslContextFactory();
			sslContextFactory.setKeyStorePath(extProps.getString("keystore.path"));
			sslContextFactory.setKeyStorePassword(extProps.getString("keystore.store-password*"));
			sslContextFactory.setKeyManagerPassword(extProps.getString("keystore.manager-password*"));
			sslContextFactory.setSslSessionTimeout(extProps.getInt("ssl-session-timeout", 30000));

			// setup authentication
			boolean needAuth = extProps.getBool("need-client-auth");
			sslContextFactory.setNeedClientAuth(needAuth);
			if (needAuth) {
				sslContextFactory.setTrustStorePath(extProps.getString("truststore.path"));
				sslContextFactory.setTrustStorePassword(extProps.getString("truststore.password*"));
			}

			connector = new ServerConnector(server, acceptors, selectors, new SslConnectionFactory(sslContextFactory, "http/1.1"), new HttpConnectionFactory(httpConf));

		} else {
			connector = new ServerConnector(server, acceptors, selectors, new HttpConnectionFactory(httpConf));
		}

		connector.setName(extProps.getString("name"));
		connector.setPort(port);
		connector.setIdleTimeout(extProps.getInt("max-idle-time"));

		return connector;
	}
}
