package tech.espublico.pades.server.rest;

import static spark.Spark.before;
import static spark.Spark.get;
import static spark.Spark.path;
import static spark.Spark.post;

import java.io.OutputStream;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.StandardCopyOption;
import java.util.Base64;

import javax.servlet.ServletRequestWrapper;

import org.eclipse.jetty.http.HttpStatus;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import spark.Request;
import spark.Response;
import spark.Route;
import spark.Spark;
import spark.embeddedserver.EmbeddedServers;
import spark.embeddedserver.EmbeddedServers.Identifiers;
import spark.embeddedserver.jetty.EmbeddedJettyFactory;
import tech.espublico.pades.server.di.Service;
import tech.espublico.pades.server.di.ServiceLocator;
import tech.espublico.pades.server.exceptions.OpenSignatureException;
import tech.espublico.pades.server.models.PDFSignedDocumentModel;
import tech.espublico.pades.server.models.PDFSignerModel;
import tech.espublico.pades.server.rest.auth.AuthService;
import tech.espublico.pades.server.rest.auth.PostAuthFilter;
import tech.espublico.pades.server.rest.auth.SimpleAuthFilter;
import tech.espublico.pades.server.rest.jetty.JettyServerFactory;
import tech.espublico.pades.server.rest.request.SignRequest;
import tech.espublico.pades.server.rest.request.SignerRequest;
import tech.espublico.pades.server.rest.request.SignerRequest.RequestOf;
import tech.espublico.pades.server.rest.results.ErrorResult;
import tech.espublico.pades.server.rest.results.SignerResponse;
import tech.espublico.pades.server.rest.validation.ValidationRoutesService;
import tech.espublico.pades.server.services.GsonService;
import tech.espublico.pades.server.services.PropertyValidatorException;
import tech.espublico.pades.server.services.pades.PadesService;
import tech.espublico.pades.server.services.storage.FilesService;

@Service
public class RoutesService {

	private static final Logger log = LoggerFactory.getLogger(RoutesService.class);

	public static RoutesService instance() {
		return ServiceLocator.INSTANCE.getInstance(RoutesService.class);
	}

	private final AuthService authService;
	private final FilesService filesService;
	private final GsonService gsonService;

	public RoutesService() throws PropertyValidatorException {
		this.filesService = FilesService.instance();
		this.authService = AuthService.instance();
		this.gsonService = GsonService.instance();

		EmbeddedServers.add(Identifiers.JETTY, new EmbeddedJettyFactory(new JettyServerFactory()));
	}

	public void start() {
		Spark.init();
		Spark.awaitInitialization();
		this.register();
	}

	public void stop() {
		Spark.awaitStop();
	}

	public void register() {
		Spark.exception(Exception.class, (e, request, response) -> {
			log.error("Generic error in [{}] {}", request.url(), e);
			SignerResponse signerResponse = new SignerResponse();
			signerResponse.setErrorResult(new ErrorResult(e));
			response.status(HttpStatus.BAD_REQUEST_400);
			response.body(this.gsonService.toJson(signerResponse));
		});

		Spark.exception(OpenSignatureException.class, (e, request, response) -> {
			log.error("Error in [{}] {}", request.url(), e);
			SignerResponse signerResponse = new SignerResponse();
			signerResponse.setErrorResult(new ErrorResult(e));
			response.status(HttpStatus.BAD_REQUEST_400);
			response.body(this.gsonService.toJson(signerResponse));
		});

		if (log.isTraceEnabled()) {
			before("/*", (request, response) -> {
				log.trace("SignerRequest [{}] [{}] [{}]", request.raw().getMethod(), request.url(), String.join(",", request.headers()));
			});
		}

		get("/status", new StatusRoute());

		before("/session", new SimpleAuthFilter());
		post("/session", (request, response) -> {
			String sessionId = request.headers(PadesServerHeaders.SESSION_ID_HEADER);
			log.info("Create session {}", sessionId);

			PadesService.instance().createSession(sessionId);

			response.status(HttpStatus.CREATED_201);
			return "{}";
		});

		before("/download", new SimpleAuthFilter());
		get("/download/:fileId", (request, response) -> {
			String stringPath = new String(Base64.getDecoder().decode(request.params(":fileId")));
			Path path = Paths.get(stringPath);

			try (OutputStream outputStream = response.raw().getOutputStream()) {
				Files.copy(path, outputStream);
				outputStream.flush();
			}

			response.status(HttpStatus.OK_200);
			return response.raw();
		});

		before("/upload", new SimpleAuthFilter());
		post("/upload", (request, response) -> {
			String sessionId = request.headers(PadesServerHeaders.SESSION_ID_HEADER);
			Path tempPath = this.filesService.createTempFile();
			PadesService.instance().addTempFile(sessionId, tempPath);
			Files.copy(((ServletRequestWrapper) request.raw()).getRequest().getInputStream(), tempPath, StandardCopyOption.REPLACE_EXISTING);

			this.authService.validate(request, tempPath);

			SignerResponse<Path> signerResponse = new SignerResponse<>();
			signerResponse.setModel(tempPath);

			response.status(HttpStatus.OK_200);
			return this.gsonService.toJson(signerResponse);
		});

		this.registerPost("/prepare", new PadesRoute<PDFSignerModel, Path>(PDFSignerModel.class) {
			@Override
			public Path handle(String sessionId, PDFSignerModel request) throws OpenSignatureException {
				return PadesService.instance().prepare(sessionId, request);
			}
		});

		this.registerPost("/digest", new PadesRoute<PDFSignerModel, byte[]>(PDFSignerModel.class) {
			@Override
			public byte[] handle(String sessionId, PDFSignerModel request) throws OpenSignatureException {
				return PadesService.instance().digest(sessionId, request);
			}
		});

		this.registerPost("/sign", new PadesRoute<SignRequest, PDFSignedDocumentModel>(SignRequest.class) {
			@Override
			public PDFSignedDocumentModel handle(String sessionId, SignRequest signRequest) throws OpenSignatureException {
				PDFSignerModel pdfSignerModel = signRequest.getPdfSignerModel();
				byte[] signedBytes = signRequest.getSignatureBytes();
				boolean requestTimeStamp = signRequest.isRequestTimeStamp();
				return PadesService.instance().sign(sessionId, pdfSignerModel, signedBytes, requestTimeStamp);
			}
		});

		ValidationRoutesService.instance().register();
	}

	private void registerPost(String path, Route route) {
		PostAuthFilter postAuthFilter = new PostAuthFilter();
		before(path, new SimpleAuthFilter());
		path(path, () -> {
			before("", postAuthFilter);
			post("", route);
			before("/", postAuthFilter);
			post("/", route);
			before("/*", postAuthFilter);
			post("/*", route);
		});
	}

}

