/**
 *
 */
package tech.espublico.pades.server.helper;

import java.io.BufferedInputStream;
import java.io.BufferedOutputStream;
import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.Serializable;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class SmartOutputStream extends OutputStream implements Serializable {

	private static final long serialVersionUID = 1L;

	private static final Logger log = LoggerFactory.getLogger(SmartOutputStream.class);

	private final static long MAX_MEM = 16384; // (16kb)

	private long size = 0;

	protected OutputStream wrappedOutputStream;

	protected File tempFile = null;

	private final File directory;

	public SmartOutputStream(File directory) {
		this.directory = directory;
		wrappedOutputStream = new ByteArrayOutputStream();
	}

	public SmartOutputStream(long expectedSize, File directory) throws IOException {
		this.directory = directory;
		if (expectedSize > MAX_MEM)
			wrappedOutputStream = createTempOutputStream();
		else
			wrappedOutputStream = new ByteArrayOutputStream((int) expectedSize);
	}

	@Override
	public void write(byte[] b) throws IOException {
		int len = b.length;
		if ((size + len) > MAX_MEM)
			swapOutputStream();

		super.write(b);
	}

	@Override
	public void write(byte[] b, int off, int len) throws IOException {
		if ((size + len) > MAX_MEM)
			swapOutputStream();

		super.write(b, off, len);
	}

	@Override
	public void write(int value) throws IOException {
		if (size == MAX_MEM) {
			swapOutputStream();
		}

		wrappedOutputStream.write(value);

		size++;
	}

	public InputStream getInputStream() throws IOException {
		this.close();

		if (wrappedOutputStream instanceof ByteArrayOutputStream) {
			return new ByteArrayInputStream(((ByteArrayOutputStream) wrappedOutputStream).toByteArray());
		} else {
			if (tempFile == null || !tempFile.exists())
				throw new IOException("Cannot create input stream from a non existent file! " + "REVIEW SmartOutputStream");

			return new BufferedInputStream(FileChannelHelper.newInputStream(tempFile), BufferUtils.BUFFERED_SIZE);
		}
	}

	public long size() {
		return size;
	}

	private OutputStream createTempOutputStream() throws IOException {
		if (tempFile == null)
			tempFile = File.createTempFile("smartOS", ".tmp", this.directory);
		return new BufferedOutputStream(FileChannelHelper.newOutputStream(tempFile), BufferUtils.BUFFERED_SIZE);
	}

	private void swapOutputStream() throws IOException {
		if (!(wrappedOutputStream instanceof ByteArrayOutputStream))
			return;

		OutputStream os = createTempOutputStream();

		os.write(((ByteArrayOutputStream) wrappedOutputStream).toByteArray());
		IOHelper.closeQuietly(wrappedOutputStream);

		wrappedOutputStream = os;

	}

	@Override
	public void close() throws IOException {
		IOHelper.closeQuietly(wrappedOutputStream);
	}
}
