package tech.espublico.pades.server.helper;

import java.io.BufferedInputStream;
import java.io.BufferedOutputStream;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.nio.file.Files;
import java.nio.file.OpenOption;
import java.nio.file.Path;

/**
 * Wrapper del los métodos JDK {@link Files#newInputStream(Path, OpenOption...)} y * {@link Files#newOutputStream(Path, OpenOption...)}.
 * <p>
 * En los {@link InputStream} devueltos, el método {@link InputStream#available()} devuelve el número de bytes reales y no una aproximación como se indica en la
 * implementación de la JDK.
 */
public class FileChannelHelper {
	/**
	 * Aplicar la configuración por defecto de la JDK en los métodos {@link Files#newInputStream(Path, OpenOption...)} y
	 * {@link Files#newOutputStream(Path, OpenOption...)}
	 */
	private static final OpenOption[] DEFAULT_OPEN_OPTIONS = {};

	/**
	 * Obtención de {@link InputStream} desde una ruta a un fichero ({@link String}) usando las opciones de apertura por defecto
	 *
	 * @param fileName
	 * 		pathname string to open
	 *
	 * @since 1.0.0
	 */
	public static InputStream newInputStream(String fileName) throws IOException {
		CheckArgument.notNull("fileName", fileName);

		return newInputStream(fileName, DEFAULT_OPEN_OPTIONS);
	}

	/**
	 * Obtención de {@link InputStream} desde una ruta a un fichero ({@link String}) indicando las opciones de apertura
	 *
	 * @param fileName
	 * 		pathname string to open
	 * @param options
	 * 		specifying how the file is opened
	 *
	 * @since 1.0.3
	 */
	public static InputStream newInputStream(String fileName, OpenOption... options) throws IOException {
		CheckArgument.notNull("fileName", fileName);
		CheckArgument.notNull("options", options);

		File file = new File(fileName);
		return newInputStream(file, options);
	}

	/**
	 * Obtención de {@link InputStream} desde un ({@link File}) usando las opciones de apertura por defecto
	 *
	 * @param file
	 * 		file to open
	 *
	 * @since 1.0.0
	 */
	public static InputStream newInputStream(File file) throws IOException {
		CheckArgument.notNull("file", file);

		return newInputStream(file, DEFAULT_OPEN_OPTIONS);
	}

	/**
	 * Obtención de {@link InputStream} desde un ({@link File}) indicando las opciones de apertura
	 *
	 * @param file
	 * 		file to open
	 * @param options
	 * 		specifying how the file is opened
	 *
	 * @since 1.0.3
	 */
	public static InputStream newInputStream(File file, OpenOption... options) throws IOException {
		CheckArgument.notNull("file", file);
		CheckArgument.notNull("options", options);

		return newInputStream(file.toPath(), options);
	}

	private static InputStream newInputStream(Path path, OpenOption... options) throws IOException {
		// In case that the file does not exists, we throw a FileNotFoundException -> This is for recompatibility with the old way os opening files for read
		if (Files.notExists(path))
			throw new FileNotFoundException();

		long size = Files.size(path);

		return new InputStreamWithAvailable(Files.newInputStream(path, options), size);
	}

	/**
	 * Obtención de {@link InputStream} con buffer de lectura desde una ruta a un fichero ({@link String}) indicando las opciones de apertura
	 *
	 * @param fileName
	 * 		pathname string to open
	 * @param options
	 * 		specifying how the file is opened
	 *
	 * @since 1.0.3
	 */
	public static InputStream newBufferedInputStream(String fileName, OpenOption... options) throws IOException {
		CheckArgument.notNull("fileName", fileName);
		CheckArgument.notNull("options", options);

		return newBufferedInputStream(new File(fileName), options);
	}

	/**
	 * Obtención de {@link InputStream} con buffer de lectura desde un ({@link File}) indicando las opciones de apertura
	 *
	 * @param file
	 * 		file to open
	 * @param options
	 * 		specifying how the file is opened
	 *
	 * @since 1.0.3
	 */
	public static InputStream newBufferedInputStream(File file, OpenOption... options) throws IOException {
		CheckArgument.notNull("file", file);
		CheckArgument.notNull("options", options);

		return newBufferedInputStream(file.toPath(), options);
	}

	/**
	 * Obtención de {@link InputStream} con buffer de lectura desde un ({@link Path}) indicando las opciones de apertura
	 *
	 * @param path
	 * 		the path to the file to open or create
	 * @param options
	 * 		specifying how the file is opened
	 *
	 * @since 1.0.3
	 */
	public static InputStream newBufferedInputStream(Path path, OpenOption... options) throws IOException {
		CheckArgument.notNull("path", path);
		CheckArgument.notNull("options", options);

		return new BufferedInputStream(FileChannelHelper.newInputStream(path, options));
	}

	/**
	 * Obtención de {@link OutputStream} desde una ruta a un fichero ({@link String})
	 *
	 * @param fileName
	 * 		pathname string to open
	 *
	 * @since 1.0.0
	 */
	public static OutputStream newOutputStream(String fileName) throws IOException {
		CheckArgument.notNull("fileName", fileName);

		return newOutputStream(fileName, DEFAULT_OPEN_OPTIONS);
	}

	/**
	 * Obtención de {@link OutputStream} desde una ruta a un fichero ({@link String}) indicando las opciones de apertura
	 *
	 * @param fileName
	 * 		pathname string to open
	 * @param options
	 * 		specifying how the file is opened
	 *
	 * @since 1.0.3
	 */
	public static OutputStream newOutputStream(String fileName, OpenOption... options) throws IOException {
		CheckArgument.notNull("fileName", fileName);
		CheckArgument.notNull("options", options);

		File file = new File(fileName);
		return newOutputStream(file, options);
	}

	/**
	 * Obtención de {@link OutputStream} desde un ({@link File}) usando las opciones de apertura por defecto
	 *
	 * @param file
	 * 		file to open
	 *
	 * @since 1.0.0
	 */
	public static OutputStream newOutputStream(File file) throws IOException {
		CheckArgument.notNull("file", file);

		return newOutputStream(file, DEFAULT_OPEN_OPTIONS);
	}

	/**
	 * Obtención de {@link OutputStream} desde un ({@link File}) indicando las opciones de apertura
	 *
	 * @param file
	 * 		file to open
	 * @param options
	 * 		specifying how the file is opened
	 *
	 * @since 1.0.3
	 */
	public static OutputStream newOutputStream(File file, OpenOption... options) throws IOException {
		CheckArgument.notNull("file", file);
		CheckArgument.notNull("options", options);

		return Files.newOutputStream(file.toPath(), options);
	}

	/**
	 * Obtención de {@link InputStream} con buffer de escritura desde una ruta a un fichero ({@link String}) indicando las opciones de apertura
	 *
	 * @param fileName
	 * 		pathname string to open or create
	 * @param options
	 * 		specifying how the file is opened
	 *
	 * @since 1.0.3
	 */
	public static OutputStream newBufferedOutputStream(String fileName, OpenOption... options) throws IOException {
		CheckArgument.notNull("fileName", fileName);
		CheckArgument.notNull("options", options);

		return newBufferedOutputStream(new File(fileName), options);
	}

	/**
	 * Obtención de {@link InputStream} con buffer de escritura desde un ({@link File}) indicando las opciones de apertura
	 *
	 * @param file
	 * 		file to open or create
	 * @param options
	 * 		specifying how the file is opened
	 *
	 * @since 1.0.3
	 */
	public static OutputStream newBufferedOutputStream(File file, OpenOption... options) throws IOException {
		CheckArgument.notNull("file", file);
		CheckArgument.notNull("options", options);

		return newBufferedOutputStream(file.toPath(), options);
	}

	/**
	 * Obtención de {@link InputStream} con buffer de escritura desde un ({@link Path}) indicando las opciones de apertura
	 *
	 * @param path
	 * 		the path to the file to open or create
	 * @param options
	 * 		specifying how the file is opened
	 *
	 * @since 1.0.3
	 */
	public static OutputStream newBufferedOutputStream(Path path, OpenOption... options) throws IOException {
		CheckArgument.notNull("path", path);
		CheckArgument.notNull("options", options);

		return new BufferedOutputStream(Files.newOutputStream(path, options));
	}

	/**
	 * Implementación de {@link InputStream} en la que el método {@link InputStream#available()} devuelve el número de bytes reales.
	 */
	private static class InputStreamWithAvailable extends InputStream {
		private InputStream is = null;
		private long available = 0;

		public InputStreamWithAvailable(InputStream is, long available) {
			this.is = is;
			this.available = available;
		}

		@Override
		public int read() throws IOException {
			int read = is.read();
			//Este metodo lee un byte de 0 a 255. Si el valor no es -1 actualizamos avaliable diciendo que hemos leido un byte
			if (read != -1)
				updateAvailable(1);
			return read;
		}

		@Override
		public int read(byte b[]) throws IOException {
			int read = is.read(b);
			updateAvailable(read);
			return read;
		}

		@Override
		public int read(byte b[], int off, int len) throws IOException {
			int read = is.read(b, off, len);
			updateAvailable(read);
			return read;
		}

		@Override
		public long skip(long n) throws IOException {
			return is.skip(n);
		}

		@Override
		public int available() throws IOException {
			return (int) available;
		}

		@Override
		public void close() throws IOException {
			is.close();
		}

		@Override
		public synchronized void mark(int readlimit) {
			is.mark(readlimit);
		}

		@Override
		public synchronized void reset() throws IOException {
			is.reset();
		}

		@Override
		public boolean markSupported() {
			return is.markSupported();
		}

		private void updateAvailable(int read) {
			available -= read;
		}
	}
}
