package tech.espublico.pades.server.di;

import java.util.Collections;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;

import tech.espublico.pades.server.di.exceptions.ServiceInstantiationException;
import tech.espublico.pades.server.di.exceptions.ServiceNotFoundException;

public enum ServiceLocator {
	INSTANCE;

	private final Map<String, ServiceEntry> serviceEntries = new ConcurrentHashMap<>();
	private final Map<String, Object> instances = new ConcurrentHashMap<>();
	private final Set<String> creating = Collections.synchronizedSet(new HashSet<>());
	private final ServiceFactory serviceFactory = new ServiceFactory();

	public void register(Class<?> serviceClass) {
		ServiceEntry serviceEntry = new ServiceEntry(serviceClass);

		if (this.serviceEntries.containsKey(serviceEntry.getName()))
			throw new IllegalArgumentException(//
					String.format("Service [%s] already registered", serviceEntry.getName()));
		if (! serviceClass.isInterface()) {
			Service service = serviceClass.getAnnotation(Service.class);
			if (service == null)
				throw new IllegalArgumentException(String.format("Service [%s] without @Service annotation", serviceEntry.getName()));
		}


		this.serviceEntries.put(serviceEntry.getName(), serviceEntry);
	}

	public void registerAll(List<Class<?>> services) {
		for (Class<?> serviceClass : services) {
			register(serviceClass);
		}
	}

	public void replace(Class<?> oldClass, Class<?> newClass) {

		if ( ! oldClass.isAssignableFrom(newClass))
			throw new IllegalArgumentException(//
					String.format("Cannot replace services are not compatible old[%s] new[%s]", //
							oldClass.getCanonicalName(), //
							newClass.getCanonicalName()));


		ServiceEntry oldService = new ServiceEntry(oldClass);
		ServiceEntry newService = new ServiceEntry(newClass);
		this.serviceEntries.put(oldService.getName(), newService);
	}

	public <T> T getInstance(String name) {
		ServiceEntry serviceEntry = this.serviceEntries.get(name);
		if (serviceEntry == null) {
			throw new ServiceNotFoundException(name, this.creating);
		}

		T object = (T) this.instances.get(name);
		if (object == null) {
			synchronized (this.creating) {
				if (this.creating.contains(name))
					throw new ServiceInstantiationException(name, this.creating);
				try {
					this.creating.add(name);
					object = serviceFactory.newInstance(serviceEntry);
					this.instances.put(name, object);
				} finally {
					this.creating.remove(name);
				}
			}
		}

		return object;
	}

	public <T> T getInstance(Class<T> serviceClass) {
		return this.getInstance(serviceClass.getCanonicalName());
	}

	public synchronized void instantiateAll() {
		for (String serviceName: this.serviceEntries.keySet()) {
			getInstance(serviceName);
		}
	}

	public synchronized void reset() {
		serviceEntries.clear();
		instances.clear();
		creating.clear();
	}
}
